% fpi_compare_stochasticss.m
% 
% FiPIt model comparison, uses closed-form decision rules in order to solve for the conditional
%   stochastic steady states associated with each regime
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
% INPUTS:   GLOBALS = structure with elements describing the model and 
%                       options for generating the results, has elements
%                       nx = number of predetermined variables
%                       ne = number of exogenous shocks
%                       ns = number of regimes
%                       nY = number of observables
%           xss     = steady state of predetermined variables
%           yss     = steady state of nonpredetermined variables
%           H       = first order transition matrix of x
%           G       = first order transition matrix of y
%           H2      = second order transition matrix of x
%           G2      = second order transition matrix of y
%           obsindex = indexes for observable variables, where
%                       the i-th observed variable is element obsindex(i)
%                       of the stacked variables [x ; y]    
%                        
% OUTPUTS:  meanstable  = (nY x ns)-matrix of means of the stochastic
%                       steady states associated with each regime, element
%                       [i,j] is the mean of the i-th variable conditional
%                       on the j-th regime      
% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
function [meanstable] = ...
    fpi_compare_stochasticss(GLOBALS,xss,yss,H,G,H2,G2,obsindex)

% -- Setup -- %
nx = GLOBALS.nx;
ny = GLOBALS.ny;
ne = GLOBALS.ne;
ns = GLOBALS.ns;
nY = GLOBALS.nY;


meanstable = zeros(nY,ns);
xf  = zeros(nx,ns);
xs  = zeros(nx,ns);
y   = zeros(ny,ns);

for ii = 1:ns
    
    % -- Mean for x -- %
    % First Order Effect
    x1 = (eye(nx)- H(:,1:nx,ii))\H(:,nx+ne+1,ii);
    state1 = [x1;zeros(ne,1);1];
    % Second Order Effect
    x2 = (eye(nx)- H(:,1:nx,ii))\(0.5*H2(:,:,ii)*kron(state1,state1));
    % Combining Effects
    xbar = xss + x1 + x2;

    % -- Mean for y -- %
    % First Order Effect
    y1 = G(:,:,ii)*[x1;zeros(ne,1);1];
    % Second Order Effect
    y2 = G(:,:,ii)*[x2;zeros(ne,1);0] + 0.5*G2(:,:,ii)*kron(state1,state1);
    % Combining Effects
    ybar = yss + y1 + y2;

    % -- Output -- %
    means = [xbar;ybar];
    xf(:,ii) = x1;
    xs(:,ii) = x2;
    y(:,ii) = ybar;
    meanstable(:,ii) = means(obsindex);
    

end

